<?php
/**
 * File Manager - Multi-Bot File Sharing System
 * Version: 3.5
 * Author: DeepSeek
 */

class FileManager {
    private $db;
    private $baseUrl;
    private $uploadDir;
    private $allowedTypes = [
        'image/jpeg', 'image/png', 'image/gif', 
        'application/pdf', 'text/plain',
        'application/msword', 
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'video/mp4', 'audio/mpeg',
        'application/zip', 'application/x-rar-compressed'
    ];
    private $maxFileSize = 20 * 1024 * 1024; // 20MB

    public function __construct() {
        $this->db = Database::getInstance();
        
        // دریافت تنظیمات از فایل پیکربندی
        require_once __DIR__ . '/../config.php';
        
        $this->baseUrl = defined('BASE_URL') ? BASE_URL : 'https://yourdomain.com';
        $this->uploadDir = defined('UPLOAD_DIR') ? UPLOAD_DIR : __DIR__ . '/../uploads/';
        
        // ایجاد پوشه آپلود اگر وجود ندارد
        if (!is_dir($this->uploadDir)) {
            mkdir($this->uploadDir, 0755, true);
        }
    }

    /**
     * ذخیره فایل در سرور و ثبت در دیتابیس
     */
    public function saveFile($sourceUrl, $originalName, $uploaderId) {
        // بررسی نوع فایل
        $fileType = $this->getRemoteFileType($sourceUrl);
        if (!$this->isFileTypeAllowed($fileType)) {
            throw new Exception("File type not allowed: $fileType");
        }

        // دریافت محتوای فایل
        $fileContent = file_get_contents($sourceUrl);
        if (!$fileContent) {
            throw new Exception("Failed to download file from Telegram");
        }

        // بررسی حجم فایل
        $fileSize = strlen($fileContent);
        if ($fileSize > $this->maxFileSize) {
            throw new Exception("File size exceeds limit (Max: " . ($this->maxFileSize / 1024 / 1024) . "MB)");
        }

        // تولید شناسه یکتا و نام فایل
        $fileId = $this->generateFileId();
        $fileExt = pathinfo($originalName, PATHINFO_EXTENSION) ?: $this->getExtensionFromMime($fileType);
        $fileName = $fileId . '.' . $fileExt;
        $filePath = $this->uploadDir . $fileName;

        // ذخیره فایل در سرور
        if (!file_put_contents($filePath, $fileContent)) {
            throw new Exception("Failed to save file to server");
        }

        // ثبت در دیتابیس
        if (!$this->db->saveFile($fileId, $originalName, $filePath, $uploaderId)) {
            unlink($filePath); // حذف فایل اگر ثبت دیتابیس ناموفق بود
            throw new Exception("Database error: " . $this->db->getLastError());
        }

        return $fileId;
    }

    /**
     * دریافت لینک دانلود فایل
     */
    public function getDownloadLink($fileId) {
        return $this->baseUrl . "/download.php?id=" . urlencode($fileId);
    }

    /**
     * دریافت مسیر فایل فیزیکی
     */
    public function getFilePath($fileId) {
        $fileInfo = $this->db->getFileById($fileId);
        return $fileInfo ? $fileInfo['file_path'] : null;
    }

    /**
     * دریافت اطلاعات فایل
     */
    public function getFileInfo($fileId) {
        return $this->db->getFileById($fileId);
    }

    /**
     * استخراج شناسه فایل از لینک
     */
    public function extractFileId($url) {
        $parts = parse_url($url);
        if (!isset($parts['query'])) return null;
        
        parse_str($parts['query'], $query);
        return $query['id'] ?? null;
    }

    /**
     * افزایش شمارنده دانلود
     */
    public function incrementDownloadCount($fileId) {
        return $this->db->incrementDownloadCount($fileId);
    }

    // ==============================
    //  توابع کمکی و امنیتی
    // ==============================
    
    private function generateFileId() {
        return bin2hex(random_bytes(16)); // 32 کاراکتر تصادفی
    }

    private function getRemoteFileType($url) {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HEADER => true,
            CURLOPT_NOBODY => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 5
        ]);
        
        $headers = curl_exec($ch);
        curl_close($ch);
        
        if (preg_match('/Content-Type:\s*([^\s;]+)/i', $headers, $matches)) {
            return strtolower(trim($matches[1]));
        }
        
        return 'application/octet-stream';
    }

    private function isFileTypeAllowed($mimeType) {
        return in_array(strtolower($mimeType), $this->allowedTypes);
    }

    private function getExtensionFromMime($mimeType) {
        $mimeMap = [
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'application/pdf' => 'pdf',
            'text/plain' => 'txt',
            'application/msword' => 'doc',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx',
            'video/mp4' => 'mp4',
            'audio/mpeg' => 'mp3',
            'application/zip' => 'zip',
            'application/x-rar-compressed' => 'rar'
        ];
        
        return $mimeMap[$mimeType] ?? 'bin';
    }

    /**
     * پاکسازی فایل‌های قدیمی (برای اجرای دوره‌ای)
     */
    public function cleanupOldFiles($days = 30) {
        $files = glob($this->uploadDir . "*");
        $deleted = 0;
        $now = time();
        
        foreach ($files as $file) {
            if (is_file($file) {
                $fileTime = filemtime($file);
                if ($now - $fileTime > $days * 86400) {
                    unlink($file);
                    $deleted++;
                }
            }
        }
        
        return $deleted;
    }

    /**
     * بررسی وجود فایل و اعتبار آن
     */
    public function validateFileId($fileId) {
        $fileInfo = $this->db->getFileById($fileId);
        
        if (!$fileInfo) {
            return "File not found";
        }
        
        if (!file_exists($fileInfo['file_path'])) {
            return "Physical file missing";
        }
        
        if ($fileInfo['expire_time'] && strtotime($fileInfo['expire_time']) < time()) {
            return "File has expired";
        }
        
        return true;
    }
}
?>